import typing as t

if t.TYPE_CHECKING:
    from gssapi.raw.named_tuples import DisplayNameResult
    from gssapi.raw.oids import OID

class Name:
    """
    A GSSAPI Name
    """

    def __new__(
        cls,
        cpy: t.Optional["Name"] = None,
    ) -> "Name": ...


def import_name(
    name: bytes,
    name_type: t.Optional["OID"] = None,
) -> Name:
    """Convert a string and a name type into a GSSAPI name.

    This method takes a string name and a name type and converts
    them into a GSSAPI :class:`Name`.

    Args:
        name (~gssapi.raw.names.Name): the string version of the name
        name_type (~gssapi.raw.types.MechType): the type of this name

    Returns:
        Name: the GSSAPI version of the name

    Raises:
        ~gssapi.exceptions.BadNameTypeError
        ~gssapi.exceptions.BadNameError
        ~gssapi.exceptions.BadMechanismError
    """


def display_name(
    name: Name,
    name_type: bool = True,
) -> "DisplayNameResult":
    """Convert a GSSAPI name into its components.

    This method converts a GSSAPI :class:`Name` back into its
    text form.  If ``name_type`` is True, it also attempts to
    retrieve the :class:`~gssapi.raw.types.NameType` of the name (otherwise the
    returned name type will be ``None``).

    Args:
        name (~gssapi.raw.names.Name): the name in question
        name_type (~gssapi.raw.types.MechType): whether or not to retrieve the
            name type

    Returns:
        DisplayNameResult: the text part of the name and its type

    Raises:
        ~gssapi.exceptions.BadNameError
    """


def compare_name(
    name1: Name,
    name2: Name,
) -> bool:
    """Check two GSSAPI names to see if they are the same.

    This method compares two GSSAPI names, checking to
    see if they are equivalent.

    Args:
        name1 (~gssapi.raw.names.Name): the first name to compare
        name2 (~gssapi.raw.names.Name): the second name to compare

    Returns:
        bool: whether or not the names are equal

    Raises:
        ~gssapi.exceptions.BadNameTypeError
        ~gssapi.exceptions.BadNameError
    """


def export_name(
    name: Name,
) -> bytes:
    """Export a GSSAPI name.

    This method "produces a canonical contigous string representation
    of a mechanism name, suitable for direct comparison for use in
    authorization functions".

    The input name must be a valid GSSAPI mechanism name, as generated by
    :func:`canonicalize_name` or
    :func:`~gssapi.raw.sec_contexts.accept_sec_context`.

    Args:
        name (~gssapi.raw.names.Name): the name to export

    Returns:
        bytes: the exported name

    Raises:
        ~gssapi.exceptions.MechanismNameRequiredError
        ~gssapi.exceptions.BadNameTypeError
        ~gssapi.exceptions.BadNameError
    """


def canonicalize_name(
    name: Name,
    mech: "OID",
) -> Name:
    """Canonicalize an arbitrary GSSAPI Name into a Mechanism Name

    This method turns any GSSAPI name into a "mechanism name" --
    a full form name specific to a mechanism.

    Args:
        name (~gssapi.raw.names.Name): the name to canonicalize
        mech (~gssapi.raw.types.MechType): the mechanism type to use to
            canonicalize the name

    Returns:
        Name: a canonicalized version of the input name

    Raises:
        ~gssapi.exceptions.BadMechanismError
        ~gssapi.exceptions.BadNameTypeError
        ~gssapi.exceptions.BadNameError
    """


def duplicate_name(
    name: Name,
) -> Name:
    """Duplicate a GSSAPI name.

    Args:
        name (~gssapi.raw.names.Name): the name to duplicate

    Returns:
        Name: a duplicate of the input name

    Raises:
        ~gssapi.exceptions.BadNameError
    """


def release_name(
    name: Name,
) -> None:
    """Release a GSSAPI name.

    This method frees a GSSAPI :class:`Name`.
    You probably won't have to do this.

    Warning:
        This method is deprecated.  Names are
        automatically freed by Python.

    Args:
        name (~gssapi.raw.names.Name): the name in question

    Raises:
        ~gssapi.exceptions.BadNameError
    """
